from unittest.mock import patch

import pytest

from apadata.api import ScraperAPI
from apadata.api.exceptions import (
    BadRequestError,
    ForbiddenRequestError,
    NotFoundRequestError,
    UnauthorizedRequestError,
    UnprocessableEntityRequestError,
)

PAYLOAD = [
    {
        "keyword": "Consulting",
        "domain": "consultingpoint.com",
        "number_of_pages": 2320,
        "content_hits": 1187,
        "title_hits": 2106,
        "url_hits": 106,
        "content_score": 10346,
        "title_score": 14436,
        "url_score": 1525,
        "created_at": "2023-10-06T09:47:45.296831Z",
        "updated_at": "2023-10-06T09:47:45.296835Z",
    }
]


class TestResponse:
    """Test Response"""

    def __init__(self, status_code, payload):
        self.status_code = status_code
        self.payload = payload

    def json(self):
        return self.payload


@pytest.mark.parametrize(
    "status_code", [200, 201, 202, 203, 204, 205, 206, 207, 208, 226]
)
@patch("apadata.api.api.requests.get")
def test_scraper_api_success(mock_api, status_code):
    mock_api.return_value = TestResponse(status_code, PAYLOAD)
    keyword = "Consulting"
    domain = "consultingpoint.com"
    frequency_features = ScraperAPI().frequency_features(keyword=keyword, domain=domain)
    assert frequency_features == [
        features for features in PAYLOAD if features["domain"] == domain
    ]


@pytest.mark.parametrize(
    "status_code,error",
    [
        (400, BadRequestError),
        (401, UnauthorizedRequestError),
        (403, ForbiddenRequestError),
        (404, NotFoundRequestError),
        (422, UnprocessableEntityRequestError),
    ],
)
@patch("apadata.api.api.requests.get")
def test_scraper_api_error(mock_api, status_code, error):
    mock_api.return_value = TestResponse(status_code, PAYLOAD)
    keyword = "Consulting"
    domain = "consultingpoint.com"
    with pytest.raises(error):
        ScraperAPI().frequency_features(keyword=keyword, domain=domain)
