import pytest

from apadata.models import (
    Acronym,
    Domain,
    ExternalIndustry,
    Keyword,
    KeywordDomainRelevance,
    RedList,
    SimilarKeywords,
    UrlTag,
)
from apadata.text_processors.text_embedder_processor import TextEmbedderProcessor
from apadata.utils import flatten


@pytest.fixture(autouse=True)
def red_list_fixture():
    RedList.objects.create(name="stupid")


@pytest.fixture(autouse=True)
def domain_fixture():
    Domain.objects.create(domain="consultingpoint.com")


@pytest.fixture(autouse=True)
def urltag_fixture():
    objects = [
        {
            "pk": 1,
            "model": "apadata.urltag",
            "fields": {
                "name": "awards-and-accolades",
                "translations": {
                    "ca": "premis i reconeixements",
                    "zh": "\u5956\u9879\u548c\u8363\u8a89",
                    "da": "priser og udm\u00e6rkelser",
                    "nl": "prijzen en onderscheidingen",
                    "en": "awards-and-accolades",
                    "fi": "palkinnot ja tunnustukset",
                    "fr": "prix et distinctions",
                    "de": "Auszeichnungen und Auszeichnungen",
                    "it": "premi e riconoscimenti",
                    "ja": "\u8cde\u3068\u6804\u8a89",
                    "ko": "\uc218\uc0c1 \ubc0f \ud45c\ucc3d",
                    "lt": "apdovanojimai ir apdovanojimai",
                    "mk": "\u043d\u0430\u0433\u0440\u0430\u0434\u0438 \u0438 \u043f\
                    u0440\u0438\u0437\u043d\u0430\u043d\u0438\u0458\u0430",
                    "nb": "priser og utmerkelser",
                    "pl": "nagrody i wyr\u00f3\u017cnienia",
                    "pt": "pr\u00eamios e elogios",
                    "ro": "premii-\u0219i-laude",
                    "es": "premios y reconocimientos",
                    "sv": "utm\u00e4rkelser och utm\u00e4rkelser",
                },
            },
        },
        {
            "pk": 2,
            "model": "apadata.urltag",
            "fields": {
                "name": "markets",
                "translations": {
                    "ca": "mercats",
                    "zh": "\u5e02\u573a",
                    "da": "markeder",
                    "nl": "markten",
                    "en": "markets",
                    "fi": "markkinoilla",
                    "fr": "march\u00e9s",
                    "de": "M\u00e4rkte",
                    "el": "\u03b1\u03b3\u03bf\u03c1\u03ad\u03c2",
                    "it": "mercati",
                    "ja": "\u5e02\u5834",
                    "ko": "\uc2dc\uc7a5",
                    "lt": "rinkose",
                    "mk": "\u043f\u0430\u0437\u0430\u0440\u0438",
                    "nb": "markeder",
                    "pl": "rynki",
                    "pt": "mercados",
                    "ro": "pie\u021belor",
                    "ru": "\u0440\u044b\u043d\u043a\u0438",
                    "es": "mercados",
                    "sv": "marknader",
                },
            },
        },
        {
            "pk": 3,
            "model": "apadata.urltag",
            "fields": {
                "name": "our-cases",
                "translations": {
                    "ca": "els nostres-casos",
                    "zh": "\u6211\u4eec\u7684\u6848\u4f8b",
                    "da": "vores-sager",
                    "nl": "onze-gevallen",
                    "en": "our-cases",
                    "fi": "meid\u00e4n tapaukset",
                    "fr": "nos-cas",
                    "de": "unsere-f\u00e4lle",
                    "it": "i nostri casi",
                    "ja": "\u79c1\u305f\u3061\u306e\u30b1\u30fc\u30b9",
                    "ko": "\uc6b0\ub9ac\uc758 \uacbd\uc6b0",
                    "lt": "m\u016bs\u0173 atvejai",
                    "nb": "v\u00e5re-saker",
                    "pl": "nasze sprawy",
                    "pt": "nossos casos",
                    "ro": "cazurile noastre",
                    "ru": "\u043d\u0430\u0448\u0438 \u0434\u0435\u043b\u0430",
                    "es": "nuestros-casos",
                    "sv": "v\u00e5ra fall",
                },
            },
        },
        {
            "pk": 9,
            "model": "apadata.urltag",
            "fields": {
                "name": "industries",
                "translations": {
                    "ca": "ind\u00fastries",
                    "zh": "\u884c\u4e1a",
                    "da": "industrier",
                    "nl": "industrie\u00ebn",
                    "en": "industries",
                    "fi": "teollisuuden aloilla",
                    "fr": "les industries",
                    "de": "Branchen",
                    "it": "industrie",
                    "ja": "\u7523\u696d",
                    "ko": "\uc0b0\uc5c5",
                    "lt": "pramon\u0117s \u0161akoms",
                    "mk": "\u0438\u043d\u0434\u0443\u0441\u0442\u0440\u0438\u0438",
                    "nb": "n\u00e6ringer",
                    "pl": "bran\u017ce",
                    "pt": "ind\u00fastrias",
                    "ro": "industrii",
                    "es": "industrias",
                    "sv": "industrier",
                },
            },
        },
    ]
    for obj in objects:
        UrlTag.objects.create(**obj["fields"])


@pytest.fixture(autouse=True)
def externalindustry_list_fixture():
    objects = [
        {
            "name": "Professional Services",
            "translations": {"en": "Professional Services"},
            "external_id": 7,
            "synonyms": {
                "en": [
                    "Consulting services",
                    "Business services",
                    "Expert services",
                    "Professional advice",
                ]
            },
        },
        {
            "name": "High Tech",
            "translations": {"en": "High Tech"},
            "external_id": 13,
            "synonyms": {
                "en": [
                    "Advanced technology",
                    "Cutting-edge innovation",
                    "High-tech sector",
                    "Technology solutions",
                ]
            },
        },
        {
            "name": "Software",
            "translations": {"en": "Software"},
            "external_id": 28,
            "synonyms": {
                "en": [
                    "Computer programs",
                    "Applications",
                    "Programs",
                    "Applications",
                    "Software solutions",
                ]
            },
        },
    ]
    for obj in objects:
        ExternalIndustry.objects.create(**obj)


@pytest.fixture(autouse=True)
def acronym_fixture():
    acronyms = [{"acronym": "BEL", "full_form": "Bharat Electronics Limited"}]
    for acronym in acronyms:
        Acronym.objects.create(
            acronym=acronym["acronym"], full_form=acronym["full_form"]
        )


@pytest.fixture(autouse=True)
def keyword_fixture():
    keywords = [
        {
            "name": "Consulting",
            "relevance_score": 0.75,
            "discarded": False,
            "similars": {},
            "discarded_reason": "",
            "en_translation": "Consulting",
            "language": "en",
            "lemma_form": "Consult",
        },
        {
            "name": "consulting",
            "relevance_score": 0.75,
            "discarded": False,
            "similars": {},
            "discarded_reason": "",
            "en_translation": "consulting",
            "language": "en",
            "lemma_form": "Consult",
        },
        {
            "name": "Consultancy",
            "relevance_score": 0.75,
            "discarded": False,
            "similars": {},
            "discarded_reason": "",
            "en_translation": "Consultancy",
            "language": "en",
            "lemma_form": "Consult",
        },
        {
            "name": "Administration",
            "relevance_score": 0.75,
            "discarded": False,
            "similars": {},
            "discarded_reason": "",
            "en_translation": "Administration",
            "language": "en",
            "lemma_form": "Administer",
        },
        {
            "name": "Verwaltung",
            "relevance_score": 0.75,
            "discarded": False,
            "similars": {},
            "discarded_reason": "",
            "en_translation": "Administration",
            "language": "de",
            "lemma_form": "Verwalt",
        },
        {
            "name": "BEL",
            "relevance_score": 0.75,
            "discarded": False,
            "similars": {},
            "discarded_reason": "",
            "en_translation": "BEL",
            "language": "en",
            "lemma_form": "Bharat Electronics Limited",
        },
        {
            "name": "Bharat Electronics Limited",
            "relevance_score": 0.75,
            "discarded": False,
            "similars": {},
            "discarded_reason": "",
            "en_translation": "Bharat Electronics Limited",
            "language": "en",
            "lemma_form": "Bharat Electronics Limited",
        },
        {
            "name": "consult",
            "relevance_score": 0.75,
            "discarded": False,
            "similars": {},
            "discarded_reason": "",
            "en_translation": "consult",
            "language": "en",
            "lemma_form": "consult",
        },
    ]
    domains = [
        "abc.com",
        "consulting.com",
        "accenture.com",
        "dw.com",
        "def.com",
        "ghi.com",
        "jkl.com",
        "mno.com",
    ]
    similar_keywords = []
    domain_objects = []
    for i, keyword in enumerate(keywords):
        relevance_score = keyword["relevance_score"]
        embedding = list(flatten(TextEmbedderProcessor(str(keyword["name"])).process()))
        keyword_object = Keyword.objects.create(
            name=keyword["name"],
            discarded=keyword["discarded"],
            discarded_reason=keyword["discarded_reason"],
            en_translation=keyword["en_translation"],
            language=keyword["language"],
            lemma_form=keyword["lemma_form"],
            embedding=embedding,
        )
        keyword_object.save()

        domain_object = Domain.objects.create(domain=domains[i])
        domain_objects.append(domain_object)
        relevance_score_object = KeywordDomainRelevance.objects.create(
            keyword=keyword_object,
            domain=domain_object,
            relevance_score=relevance_score,
        )
        relevance_score_object.save()
        similar_keywords.append(keyword_object)

    for i, keyword in enumerate(keywords):
        blank_object = similar_keywords[len(similar_keywords) - i - 1]
        similars_object = SimilarKeywords.objects.create(
            keyword1=keyword_object, keyword2=blank_object
        )
        similars_object.save()

        keyword_object.similars.add(blank_object)
        # keyword_object.relevance_score.add(domain_objects[i])
        keyword_object.save()


def pytest_collection_modifyitems(items):
    """Add django_db marker to all tests in order to use the test database."""
    for item in items:
        item.add_marker(pytest.mark.django_db(databases="__all__"))
