import json
from unittest.mock import patch

from apadata.loaders.json_loader import JSONLoader

from ...spacy.constants import CONTENTS_SEPARATOR
from ..elasticsearch_client import ElasticsearchClient
from .mocked_elasticsearch import MockES

mocked_response_accenture = JSONLoader(
    "elasticsearch_client/tests/mocked_accenture_response.json"
).load()


def assert_elasticsearch_query(index, from_, size, highlight, source_excludes):
    assert index == "content-*"
    assert from_ is None
    assert size == 1
    assert source_excludes == []
    assert highlight == {
        "number_of_fragments": 3,
        "fragment_size": 250,
        "fields": {
            "title": {
                "pre_tags": ['<span style="background-color: #fff70061">'],
                "post_tags": ["</span>"],
            },
            "content": {
                "pre_tags": ['<span style="background-color: #fff70061">'],
                "post_tags": ["</span>"],
            },
        },
    }


class MockedElasticsearch(MockES):
    """Mocked ES client that returns a mocked response for the query"""

    def search(self, index, query, from_, size, highlight, source_excludes):
        assert_elasticsearch_query(index, from_, size, highlight, source_excludes)
        assert query == {
            "bool": {
                "must": [
                    {
                        "simple_query_string": {
                            "default_operator": "or",
                            "fields": ["title", "content"],
                            "query": "accenture",
                        }
                    }
                ]
            }
        }
        return mocked_response_accenture


class MockedElasticsearchNoKeyword(MockES):
    """Mocked ES client that returns a mocked response for the query without keyword"""

    def search(self, index, query, from_, size, highlight, source_excludes):
        assert_elasticsearch_query(index, from_, size, highlight, source_excludes)
        assert query == {
            "bool": {
                "must": [
                    {"match": {"domain": "accenture.com"}},
                    {"match": {"url": "accenture.com/about"}},
                ]
            }
        }
        return mocked_response_accenture


def return_mocked_es(
    url: str,  # pylint: disable=unused-argument
) -> MockedElasticsearch:
    return MockedElasticsearch()


def return_mocked_es_no_keyword(
    url: str,  # pylint: disable=unused-argument
) -> MockedElasticsearchNoKeyword:
    return MockedElasticsearchNoKeyword()


@patch("apadata.elasticsearch_client.elasticsearch_client.elasticsearch.Elasticsearch")
def test_elasticsearch(mock_es, snapshot):
    mock_es.side_effect = return_mocked_es
    es_client = ElasticsearchClient()
    hits = es_client.search(
        domain="accenture.com", keyword="accenture", url="accenture.com/about"
    )
    assert hits
    web_contents = es_client.get_web_contents(hits)
    assert len(web_contents) == 1
    result: str = CONTENTS_SEPARATOR.join(web_contents)
    assert len(result) == 1396
    snapshot.assert_match(
        json.dumps(result[0]), "elastic_search_result_accenture_title_content"
    )


@patch("apadata.elasticsearch_client.elasticsearch_client.elasticsearch.Elasticsearch")
def test_elasticsearch_no_keyword(mock_es, snapshot):
    mock_es.side_effect = return_mocked_es_no_keyword
    es_client = ElasticsearchClient()
    hits = es_client.search(domain="accenture.com", url="accenture.com/about")
    assert hits
    web_contents = es_client.get_web_contents(hits)
    assert len(web_contents) == 1
    result = CONTENTS_SEPARATOR.join(web_contents)
    assert len(result) == 1396
    snapshot.assert_match(
        json.dumps(result[0]),
        "elastic_search_result_accenture_title_content_no_keyword",
    )
