from typing import Any, Dict, List, Union

import time

from cachetools.func import ttl_cache
from django.db import models


class ExternalIndustry(models.Model):
    """Model that contains the relation between external industries taxonomies and
    apadua industries taxonomies"""

    class Meta:
        indexes = [
            models.Index(fields=["name"]),
        ]

    name = models.CharField(max_length=100)  # english always
    translations = models.JSONField(default={})  # key=language, value=the name
    synonyms = models.JSONField(
        # Each synonym is a json with key=language and value=the translated value
        # {"en": "Semiconductor", "de": <<"Semiconductor">> in german}
        # limit to 10 synonyms per industry
        default={}
    )
    external_id = models.CharField(null=True, max_length=100)
    # refers to id in Apadua database

    def searchable_entities(self, language: str) -> Dict[str, Union[str, List[str]]]:
        translations = (
            [self.translations[language]] if language in self.translations else []
        )
        synonyms = self.synonyms[language] if language in self.synonyms else []
        return {
            "name": self.name,
            "translations": translations,
            "synonyms": synonyms,
        }

    @staticmethod
    @ttl_cache(maxsize=10_000, ttl=86_400, timer=time.monotonic, typed=False)
    def load_all() -> List[Any]:
        return list(ExternalIndustry.objects.all())
