from typing import Any, Dict

from apadata.pipelines.pipeline_context import PipelineContext
from apadata.pipelines.pipeline_module import PipelineModule
from apadata.spacy.constants import KeywordsProcessingMode
from apadata.text_processors.keywords.enrichers.case_insensitive_finder_enricher import (
    CaseInsensitiveFinderEnricher,
)
from apadata.text_processors.keywords.reducers.case_insensitive_finder_reducer import (
    CaseInsensitiveFinderReducer,
)

Input = Dict[str, Any]
Output = Dict[str, Any]


class CaseInsensitiveFinderModule(PipelineModule[Input, Output]):
    """
    Checks for keywords which are the same when doing a case-insensitive comparison,
    if there are any, then the Title case on or the UPPERCASE one will be
    prioritized in being kept and the lowercase one will be dropped
    """

    def __init__(
        self,
        name: str = "case_insensitive_finder",
        mode: str = KeywordsProcessingMode.REDUCE,
        **kwargs: Any,
    ):
        super().__init__(
            name,
            **kwargs,
        )
        self.mode = mode

    def run(self, context: PipelineContext) -> Output:
        keyword = context.search_field("keyword")
        if self.mode == KeywordsProcessingMode.REDUCE:
            CaseInsensitiveFinderReducer(keyword).process()
        elif self.mode == KeywordsProcessingMode.ENRICH:
            CaseInsensitiveFinderEnricher(keyword).process()
        else:
            raise ValueError(f"Wrong mode given: {self.mode}!")
        return dict(context.payload)
