from typing import Any, Dict

from apadata.pipelines.pipeline_context import PipelineContext
from apadata.pipelines.pipeline_module import PipelineModule
from apadata.spacy.constants import KeywordsProcessingMode
from apadata.text_processors.keywords.enrichers.synonymy_finder_enricher import (
    SynonymyFinderEnricher,
)
from apadata.text_processors.keywords.reducers.synonymy_finder_reducer import (
    SynonymyFinderReducer,
)

Input = Dict[str, Any]
Output = Dict[str, Any]


class SynonymyFinderModule(PipelineModule[Input, Output]):
    """
    An external library for synonyms is generating all the synonyms candidates for
    the given keywords and then in the same way that the similarity finder works,
    tuples of synonyms are created and only one of them from the whole tuple is
    randomly kept
    """

    def __init__(
        self,
        name: str = "synonymy_finder",
        mode: str = KeywordsProcessingMode.REDUCE,
        **kwargs: Any,
    ):
        super().__init__(
            name,
            **kwargs,
        )
        self.mode = mode

    def run(self, context: PipelineContext) -> Output:
        keyword = context.search_field("keyword")
        if self.mode == KeywordsProcessingMode.REDUCE:
            SynonymyFinderReducer(keyword).process()
        elif self.mode == KeywordsProcessingMode.ENRICH:
            SynonymyFinderEnricher(keyword).process()
        else:
            raise ValueError(f"Wrong mode given: {self.mode}!")
        return dict(context.payload)
