from typing import Any, Dict

from apadata.chatgpt.chat_gpt import ChatGpt
from apadata.pipelines.pipeline_context import PipelineContext
from apadata.pipelines.pipeline_module import PipelineModule

Input = Dict[str, Any]
Output = str


def query0(company_name: str, keyword: str, num_words: int, text: str) -> str:
    query = 'Start your summary with the words: "The company offers..." '
    query += "The given content explains the viewpoint of "
    query += f"company {company_name} "
    query += f"on the subject matter {keyword}. "
    query += "Summarize this view point. "
    query += f"Do not exceed {str(num_words)} words. "
    query += "Do not write about general company information. "
    query += "Be as specific as possible. "
    query += f"This is the given content: {text}"
    return query


def query1(company_name: str, keyword: str, num_words: int, text: str) -> str:
    query = "You are a company analyst. Analyze the below web content of "
    query += f"the company {company_name} and summarize as an example the services, "
    query += f"products, and solutions that {company_name} is offering to their "
    query += f"clients within the field of {keyword} without naming the company. "
    query += f"This is the web content: {text}. Don't be too general, "
    query += f"be specific and focus your answer on this field: {keyword}. "
    query += "Use only this given web content, "
    query += "do not refer to any other sources of information. "
    query += f"The whole summary should not exceed {str(num_words)} words."
    return query


def query1a(company_name: str, keyword: str, num_words: int, text: str) -> str:
    query = "You are a company analyst. Analyze the below web content of "
    query += f"the company {company_name} and summarize as an example the services, "
    query += f"products, and solutions that {company_name} is offering to their "
    query += f"clients within the field of {keyword} without naming the company. "
    query += f"This is the web content: {text}. Don't be too general, "
    query += f"be specific and focus your answer on this field: {keyword}. "
    query += "Use only this given web content, "
    query += "do not refer to any other sources of information. "
    query += f"The whole summary should not exceed {str(num_words)} words. "

    query += "You should write only 4 sentences. First one should "
    query += "catch attention, second one should raise interest, third one should "
    query += "create a desire and the fourth one a call to action."
    return query


def query1b(company_name: str, keyword: str, num_words: int, text: str) -> str:
    query = "You are a company analyst. Analyze the below web content of "
    query += f"the company {company_name} and summarize as an example the services, "
    query += f"products, and solutions that {company_name} is offering to their "
    query += f"clients within the field of {keyword} without naming the company. "
    query += f"This is the web content: {text}. Don't be too general, "
    query += f"be specific and focus your answer on this field: {keyword}. "
    query += "Use only this given web content, "
    query += "do not refer to any other sources of information. "
    query += f"The whole summary should not exceed {str(num_words)} words. "

    query += "You should write only 4 sentences. First one should be a statement, "
    query += "second one should provide an explanation, third one should "
    query += "give an example and the fourth one should mention an implication."
    return query


def query1c(company_name: str, keyword: str, num_words: int, text: str) -> str:
    query = "You are a company analyst. Analyze the below web content of "
    query += f"the company {company_name} and summarize as an example the services, "
    query += f"products, and solutions that {company_name} is offering to their "
    query += f"clients within the field of {keyword} without naming the company. "
    query += f"This is the web content: {text}. Don't be too general, "
    query += f"be specific and focus your answer on this field: {keyword}. "
    query += "Use only this given web content, "
    query += "do not refer to any other sources of information. "
    query += f"The whole summary should not exceed {str(num_words)} words. "

    query += "You should write only 3 sentences. First one should be a definition, "
    query += "second one should be an offering and the third one should be an example."
    return query


def query2(company_name: str, keyword: str, num_words: int, text: str) -> str:
    query = "To better understand the service and solution offering "
    query += f"of {company_name}, explain the topic {keyword} in one sentence. "
    query += f"In two more sentences, sum up the given content {text} with "
    query += f"a focus on the potential service offering of company {company_name}. "
    query += "Do not use the company name in the summary. "
    query += f"Do not exceed {str(num_words)} words."
    return query


def query3(company_name: str, keyword: str, num_words: int, text: str) -> str:
    query = f"Analyze the below web content of the company {company_name} and "
    query += f"summarize it in less than {str(num_words)} without naming the company. "
    query += "The target audience of this summary should be business people in search "
    query += f"of potential service providers in the field of {keyword}. "
    query += f"This is the web content{text}. "
    query += "Use only this given web content, do not refer to any other information."
    return query


def query4(company_name: str, keyword: str, num_words: int, text: str) -> str:
    query = f"Analyze the below web content of the company {company_name} and "
    query += f"summarize it in less than {str(num_words)} what are the services "
    query += f"related to {keyword} received from the company {company_name} "
    query += "without naming the company. The target audience of this summary "
    query += "should be business people in search of potential service providers. "
    query += f"This is the web content: {text}. Use only this given web content, "
    query += "do not refer to any other information."
    return query


class SummaryModule(PipelineModule[Input, Output]):
    """Class for instantiating ChatGpt object

    This module is designed to make chatGPT requests and generate the summary outputs.
    Metadata context variables below are used to adjust the GPT model:
    prompt_index: int - specify what query to be prompted to ChatGPT
    num_words: int - max words requested by query
    engine: Literal["gpt3.5", "gpt4"] - what gpt to use
    temperature: float - adjust GPT creativity
    max_tokens: int - max words in response
    top_p: float
    frequency_penalty: float
    presence_penalty: float
    """

    def __init__(
        self,
        name: str = "chat_gpt",
        **kwargs: Any,
    ):
        super().__init__(name, **kwargs)

    @staticmethod
    def make_query(
        company_name: str = "",
        keyword: str = "",
        num_words: int = 100,
        text: str = "",
        prompt_index: int = 0,
    ) -> str:
        query_options = [
            query0,
            query1,
            query1a,
            query1b,
            query1c,
            query2,
            query3,
            query4,
        ]

        if prompt_index < len(query_options):
            return query_options[prompt_index](company_name, keyword, num_words, text)

        return query_options[0](company_name, keyword, num_words, text)
        # 0 can be maybe changed with prompt_index%len(query_options)

    def run(self, context: PipelineContext) -> Output:
        """
        Function that runs the ChatGPT module

        @param context: Dict[str, Any] - it should contain company_name:str,
        keyword:str, num_words:int, web_content:str
        @return: output_messages: List[Dict[str, str]] - list of summaries
        """

        if not isinstance(context.payload, dict):
            raise ValueError("Payload should have been a dictionary")

        query = SummaryModule.make_query(
            keyword=str(context.search_field("keyword", "")),
            num_words=int(context.search_field("num_words", 200)),
            text=str(context.search_field("text", "")),
            prompt_index=int(context.search_field("prompt_index", 0)),
        )
        input_messages = [{"role": "system", "content": query}]

        return ChatGpt().execute_call(context, input_messages)
