from typing import Dict, Optional

from django.db import models

from apadata.models import Domain, Keyword, KeywordDomainRelevance
from apadata.pipelines.evaluator_pipeline import EvaluatorPipeline
from apadata.pipelines.pipeline_context import PipelineContext
from apadata.text_processors import LangDetectTextProcessor


def get_persisted_results(
    keyword_object: Keyword, domain_object: Domain
) -> (Optional)[float]:
    query_set = Keyword.objects.filter(name=keyword_object)
    if not query_set:
        return None
    domain_to_relevance_score = dict(query_set[0].domain_to_relevance_score)
    if domain_object.domain in domain_to_relevance_score:
        return float(domain_to_relevance_score[domain_object])
    return None


def evaluator_task(*, keyword: str, domain: str) -> float:
    # check whether we have the result already stored in the database
    try:
        keyword_object = Keyword.objects.get(name=keyword)
    except (models.ObjectDoesNotExist, Keyword.DoesNotExist):
        keyword_object = Keyword.objects.create(name=keyword)

    try:
        domain_object = Domain.objects.get(domain=domain)
    except (models.ObjectDoesNotExist, Keyword.DoesNotExist):
        domain_object = Domain.objects.create(domain=domain)

    persisted_results = get_persisted_results(keyword_object, domain_object)
    if persisted_results:
        return persisted_results

    # run the pipeline and obtain the result
    payload = {
        "keyword": keyword,
        "domain": domain,
        "language": LangDetectTextProcessor(keyword).detect(),
        "strategy": "weighted_mean",
    }
    context = PipelineContext(payload=payload)
    evaluator_pipeline = EvaluatorPipeline(context=context)
    context_result = evaluator_pipeline.run()
    result: Dict[str, float] = context_result.result
    relevance_score: float = result["relevance_score"]

    # persist the result in the database
    KeywordDomainRelevance.objects.create(
        domain=domain_object, keyword=keyword_object, relevance_score=relevance_score
    )
    keyword_object.relevance_score.set(objs=[domain_object])
    keyword_object.save()

    return relevance_score
