from typing import Any, List, Tuple

from apadata.spacy import SpacyConfiguration
from apadata.spacy.constants import LangCode, SpacyGenre, SpacyPipeline

from .. import SpacyTextProcessor
from .constants import (
    GREEN_LIST_DEPENDENCIES_TAGS_SPACY,
    GREEN_LIST_NAMED_ENTITY_RECOGNITION_TAGS_SPACY,
    GREEN_LIST_PART_OF_SPEECH_TAGS_SPACY,
    RED_LIST_DEPENDENCIES_TAGS_SPACY,
    RED_LIST_NAMED_ENTITY_RECOGNITION_TAGS_SPACY,
    RED_LIST_PART_OF_SPEECH_TAGS_SPACY,
)
from .evaluator import Evaluator

DEFAULT_PERCENTAGE = 0.5


class LinguisticalEvaluator(Evaluator):
    """Evaluates a keyword based on its part of speech, dependency trees and named
    entities"""

    def __init__(self, language: str):
        self.language = language
        self.lang_code = LangCode(language)
        self.pipeline_genre = SpacyGenre(SpacyGenre.CORE_WEB)
        self.pipeline_type = SpacyPipeline(SpacyPipeline.SM_PIPELINE)
        self.spacy_config = SpacyConfiguration(
            lang_code=self.lang_code,
            pipeline_type=self.pipeline_type,
            pipeline_genre=self.pipeline_genre,
        )

    def evaluate(self, keyword: str) -> float:
        spacy_document = self.get_spacy_document(keyword)
        return (
            self.calculate_quality_tag_percentage(
                self.get_pos_tags_and_lists(spacy_document)
            )
            + self.calculate_quality_tag_percentage(
                self.get_ner_tags_and_lists(spacy_document)
            )
            + self.calculate_quality_tag_percentage(
                self.get_dep_tags_and_lists(spacy_document)
            )
        ) / 3

    def get_pos_tags_and_lists(
        self, spacy_document: Any
    ) -> (Tuple)[List[str], List[str], List[str]]:
        return (
            [token.pos_ for token in spacy_document],
            GREEN_LIST_PART_OF_SPEECH_TAGS_SPACY,
            RED_LIST_PART_OF_SPEECH_TAGS_SPACY,
        )

    def get_ner_tags_and_lists(
        self, spacy_document: Any
    ) -> (Tuple)[List[str], List[str], List[str]]:
        return (
            [token.label_ for token in spacy_document.ents],
            GREEN_LIST_NAMED_ENTITY_RECOGNITION_TAGS_SPACY,
            RED_LIST_NAMED_ENTITY_RECOGNITION_TAGS_SPACY,
        )

    def get_dep_tags_and_lists(
        self, spacy_document: Any
    ) -> (Tuple)[List[str], List[str], List[str]]:
        return (
            [token.dep_ for token in spacy_document],
            GREEN_LIST_DEPENDENCIES_TAGS_SPACY,
            RED_LIST_DEPENDENCIES_TAGS_SPACY,
        )

    def calculate_quality_tag_percentage(
        self, tags_and_lists: Tuple[List[str], List[str], List[str]]
    ) -> float:
        tags, green_list, red_list = tags_and_lists
        num_good_tags = 0
        for tag in tags:
            if tag in green_list and tag not in red_list:
                num_good_tags += 1
        return num_good_tags / len(tags) if len(tags) else DEFAULT_PERCENTAGE

    def get_spacy_document(
        self,
        keyword: str,
    ) -> Any:
        spacy_text_processor: SpacyTextProcessor = SpacyTextProcessor(
            keyword, self.spacy_config
        )
        spacy_document = spacy_text_processor.obtain_doc()
        return spacy_document
