from typing import Set, Union

from django.db import models
from nltk.corpus import wordnet

from apadata.models import Keyword
from apadata.text_processors import TextProcessor


class SynonymyFinder(TextProcessor):
    """Finds synonyms of a word, if any of those synonyms are already present in the
    database as different entries, then they are discarded"""

    def __init__(self, text: str):
        super().__init__(text)

    def process(self) -> Union[str, Set[str]]:
        try:
            keyword_obj = Keyword.objects.get(name=self.text)
        except (models.ObjectDoesNotExist, Keyword.DoesNotExist):
            return self.text
        if keyword_obj.discarded:
            return self.text
        return set(
            {
                lemma.name()
                for syn in wordnet.synsets(self.text)
                for lemma in syn.lemmas()
            }
        )
