from typing import Tuple

import urllib
from string import ascii_letters, ascii_lowercase, ascii_uppercase

import bs4
import goslate
import timeout_decorator
from deep_translator import GoogleTranslator
from timeout_decorator.timeout_decorator import TimeoutError as TimeoutDecoratorError
from translate import Translator

from apadata.text_processors import TextProcessor

NUM_SECONDS_TIMEOUT = 5
GOSLATE_TRANSLATOR: goslate.Goslate = goslate.Goslate()


class TextTranslator(TextProcessor):
    """
    Translates a text from a given source_language to a destination_language
    """

    def __init__(
        self, text: str, source_language: str = "en", destination_language: str = "de"
    ):
        super().__init__(text)
        self.source_language = source_language
        self.destination_language = destination_language

    @staticmethod
    def is_translatable(text: str) -> bool:
        for letter in text:
            if (
                letter in ascii_letters
                or letter in ascii_lowercase
                or letter in ascii_uppercase
                or letter.lower() in "üöä"
            ):
                return True
        return False

    @timeout_decorator.timeout(NUM_SECONDS_TIMEOUT)
    def translate_goslate(self) -> str:
        translation: str = GOSLATE_TRANSLATOR.translate(
            self.text, self.destination_language, self.source_language
        )
        return translation

    @timeout_decorator.timeout(NUM_SECONDS_TIMEOUT)
    def translate_google(self) -> str:
        lang_replacement_dict = {"zh": "zh-CN", "nb": "no"}
        source_language = (
            self.source_language
            if (self.source_language not in lang_replacement_dict)
            else lang_replacement_dict[self.source_language]
        )
        destination_language = (
            self.destination_language
            if (self.destination_language not in lang_replacement_dict)
            else lang_replacement_dict[self.destination_language]
        )
        google_translator = GoogleTranslator(
            source=source_language, target=destination_language
        )
        translation: str = google_translator.translate(self.text)
        return translation

    @timeout_decorator.timeout(NUM_SECONDS_TIMEOUT)
    def translate_translator(self) -> str:
        _translator = Translator(
            to_lang=self.destination_language,
            from_lang=self.source_language,
        )
        translation: str = _translator.translate(self.text)
        return translation

    @staticmethod
    def translation_sanity_check(translation: str) -> bool:
        return (
            isinstance(translation, str)
            and len(translation) > 0
            and not (
                translation.startswith(
                    "MYMEMORY WARNING: YOU USED ALL AVAILABLE "
                    "FREE TRANSLATIONS FOR TODAY. NEXT AVAILABLE "
                    "IN"
                )
            )
        )

    def process(self) -> Tuple[str, bool]:
        return self.translate_text()

    def translate_text(self) -> Tuple[str, bool]:
        translation = self.text
        is_translated = False
        if TextTranslator.is_translatable(self.text):
            try:
                translation = self.translate_goslate()
                is_translated = TextTranslator.translation_sanity_check(translation)
            except (
                ValueError,
                urllib.error.HTTPError,
                TimeoutDecoratorError,
                IndexError,
            ):
                pass

            if not is_translated:
                try:
                    translation = self.translate_google()
                    is_translated = TextTranslator.translation_sanity_check(translation)
                except (
                    NameError,
                    TimeoutDecoratorError,
                    bs4.builder.ParserRejectedMarkup,
                ):
                    pass

            if not is_translated:
                try:
                    translation = self.translate_translator()
                    is_translated = TextTranslator.translation_sanity_check(translation)
                except (ValueError, TimeoutDecoratorError):
                    pass

        if not is_translated:
            translation = self.text
        return translation, is_translated
