from typing import Any, Optional

import pinecone

from apadata.constants import PINECONE_API_KEY, PINECONE_ENVIRONMENT
from apadata.utils import Singleton


class VectorDBAPI(metaclass=Singleton):
    """
    Class that gets requests results from Pinecone's Vector DB
    """

    def __init__(
        self,
        index_name: str = "main",
        dimension: int = 384,
        metric: str = "cosine",
        renew_index: bool = False,
    ):
        self.set_pinecone_api()
        self.index_name = index_name
        self.renew_index = renew_index
        self.metric = metric
        self.dimension = dimension
        if self.index_name not in pinecone.list_indexes():
            self.create_index()
        elif renew_index:
            self.delete_index()
            self.create_index()
        self.index = pinecone.Index(index_name=index_name)

    @staticmethod
    def set_pinecone_api():
        pinecone.init(api_key=PINECONE_API_KEY, environment=PINECONE_ENVIRONMENT)

    def create_index(self) -> Any:
        return pinecone.create_index(
            self.index_name, dimension=self.dimension, metric=self.metric
        )

    def insert(self, text: str, embedding: Any) -> Any:
        self.index.upsert([(text, embedding)])
        return embedding

    def query(self, embedding: Any, top_k: int = 3) -> Any:
        return self.index.query(vector=embedding, top_k=top_k)

    def delete_index(self) -> Any:
        return pinecone.delete_index(self.index_name)
