from typing import Any, Dict, List, Optional, Union

import abc
from urllib.parse import urljoin

import requests
from requests.models import Response

from .exceptions import (
    BadRequestError,
    ForbiddenRequestError,
    NotFoundRequestError,
    UnauthorizedRequestError,
    UnprocessableEntityRequestError,
)

Payload = Union[Dict[str, Any], List[Any]]


class API(abc.ABC):
    """
    Abstract general class for the API

    Parameters
    ----------
    headers : Dict[str, str]
        Contains optional arguments that will be transmitted in the request
    api_url : str
        Url of the api we are sending requests to
    timeout : int
        Number of seconds until halting a request
    """

    headers: Dict[str, str]
    api_url: str
    timeout: int

    def __init__(
        self, api_url: str, headers: Optional[Dict[str, str]], timeout: int = 10_000
    ):
        self.api_url = api_url
        self.headers = headers or {}
        self.timeout = timeout

    def get(
        self,
        endpoint: str,
        query_params: Optional[Dict[str, str]] = None,
        api_url: str = "",
        options: Optional[Dict[str, Any]] = None,
    ) -> Response:
        response = requests.get(
            urljoin(api_url or self.api_url, endpoint),
            headers=self.get_headers(options),
            params=query_params,
            timeout=self.timeout,
        )

        return self.handle_response(response)

    def post(
        self,
        endpoint: str,
        payload: Payload,
        query_params: Optional[Dict[str, str]] = None,
        api_url: str = "",
        options: Optional[Dict[str, Any]] = None,
    ) -> Response:
        response = requests.post(
            urljoin(api_url or self.api_url, endpoint),
            json=payload,
            params=query_params,
            headers=self.get_headers(options),
            timeout=options.get("timeout", self.timeout) if options else self.timeout,
        )

        return self.handle_response(response)

    def put(
        self,
        endpoint: str,
        payload: Payload,
        query_params: Optional[Dict[str, str]] = None,
        api_url: str = "",
        options: Optional[Dict[str, Any]] = None,
    ) -> Response:
        response = requests.put(
            urljoin(api_url or self.api_url, endpoint),
            json=payload,
            params=query_params,
            headers=self.get_headers(options),
            timeout=options.get("timeout", self.timeout) if options else self.timeout,
        )

        return self.handle_response(response)

    def delete(
        self,
        endpoint: str,
        payload: Payload,
        query_params: Optional[Dict[str, str]] = None,
        api_url: str = "",
        options: Optional[Dict[str, Any]] = None,
    ) -> Response:
        response = requests.delete(
            urljoin(api_url or self.api_url, endpoint),
            json=payload,
            params=query_params,
            headers=self.get_headers(options),
            timeout=options.get("timeout", self.timeout) if options else self.timeout,
        )

        return self.handle_response(response)

    def handle_response(self, response: Response) -> Response:
        if 200 <= response.status_code <= 299:
            return response

        if 400 <= response.status_code <= 499:
            return self.handle_error(response)

        return response

    def handle_error(self, response: Response) -> Response:
        status = response.status_code
        if status == 400:
            raise BadRequestError("Bad Request", response=response)
        if status == 401:
            raise UnauthorizedRequestError("Unauthorized", response=response)
        if status == 403:
            raise ForbiddenRequestError("Forbidden", response=response)
        if status == 404:
            raise NotFoundRequestError("Not Found", response=response)
        if status == 422:
            raise UnprocessableEntityRequestError("Not Valid", response=response)

        return response

    def get_headers(self, options: Optional[Dict[str, Any]] = None) -> Dict[str, str]:
        options = options or {}
        return {**self.headers, **options.get("headers", {})}
