from typing import Any, Optional

from json import JSONDecodeError

import structlog

from apadata.api.api import API
from apadata.constants import CLEARBIT_API_KEY

logger = structlog.get_logger(__name__)


class ClearbitAPI(API):
    """
    Abstract general class for the Clearbit API
    """

    def __init__(self):
        super().__init__(
            api_url="https://company.clearbit.com",
            headers={"Authorization": f"Bearer {CLEARBIT_API_KEY}"},
        )

    def industry(self, domain: str) -> Any:
        """
        Receives a domain of a company and returns the industry in which the company
         is active
        """
        endpoint = "/v2/companies/find"
        params = {"domain": domain}
        response = self.get(endpoint=endpoint, query_params=params)
        try:
            response_json = response.json()
        except JSONDecodeError as e:
            logger.error("Error decoding JSON", error=e)
            return None

        return response_json.get("category", {}).get("industry", None)

    def suggest(self, name: str, count: Optional[int] = None) -> Any:
        """
        Receives a name of a company and suggests a domain for that company name
        """
        url = "https://autocomplete.clearbit.com"
        endpoint = "/v1/companies/suggest"
        query_params = {"query": name}
        response = self.get(endpoint=endpoint, query_params=query_params, api_url=url)
        try:
            response_json = response.json()
        except JSONDecodeError as e:
            logger.error("Error decoding JSON", error=e)
            return None

        domains = [suggestion.get("domain") for suggestion in response_json]
        return domains[:count] if count else domains
