from unittest.mock import patch

import pytest

from apadata.api.clearbit.clearbit_api import ClearbitAPI
from apadata.api.exceptions import (
    BadRequestError,
    ForbiddenRequestError,
    NotFoundRequestError,
    UnauthorizedRequestError,
    UnprocessableEntityRequestError,
)


class TestResponse:
    """Test Response"""

    def __init__(self, status_code, payload):
        self.status_code = status_code
        self.payload = payload

    def json(self):
        return self.payload


@pytest.mark.parametrize(
    "status_code", [200, 201, 202, 203, 204, 205, 206, 207, 208, 226]
)
@patch("apadata.api.api.requests.get")
def test_clearbit_api_domain_to_industry_success(mock_requests, status_code):
    mock_requests.return_value = TestResponse(
        status_code, {"category": {"industry": "IT"}}
    )
    domain = "kpmg.com"
    industry = ClearbitAPI().industry(domain=domain)

    assert industry == "IT"


@pytest.mark.parametrize(
    "status_code,error",
    [
        (400, BadRequestError),
        (401, UnauthorizedRequestError),
        (403, ForbiddenRequestError),
        (404, NotFoundRequestError),
        (422, UnprocessableEntityRequestError),
    ],
)
@patch("apadata.api.api.requests.get")
def test_clearbit_api_domain_to_industry_error(mock_requests, status_code, error):
    mock_requests.return_value = TestResponse(
        status_code, {"category": {"industry": "IT"}}
    )
    domain = "kpmg.com"

    with pytest.raises(error):
        ClearbitAPI().industry(domain=domain)


@pytest.mark.parametrize(
    "status_code", [200, 201, 202, 203, 204, 205, 206, 207, 208, 226]
)
@patch("apadata.api.api.requests.get")
def test_clearbit_api_name_to_domain_success(mock_requests, status_code):
    mock_requests.return_value = TestResponse(
        status_code, [{"domain": "kpmg.us", "name": "KPMG USA"}]
    )
    name = "KPMG"
    count = 1
    domains = ClearbitAPI().suggest(name=name, count=count)

    assert domains[count - 1] == "kpmg.us"


@pytest.mark.parametrize(
    "status_code,error",
    [
        (400, BadRequestError),
        (401, UnauthorizedRequestError),
        (403, ForbiddenRequestError),
        (404, NotFoundRequestError),
        (422, UnprocessableEntityRequestError),
    ],
)
@patch("apadata.api.api.requests.get")
def test_clearbit_api_name_to_domain_error(mock_requests, status_code, error):
    mock_requests.return_value = TestResponse(
        status_code, [{"domain": "kpmg.us", "name": "KPMG USA"}]
    )
    name = "KPMG"
    count = 1
    with pytest.raises(error):
        ClearbitAPI().suggest(name=name, count=count)
