from typing import Optional

import json
from dataclasses import dataclass
from unittest.mock import patch

from lxml.etree import XMLSyntaxError
from requests.exceptions import RequestException

from apadata.loaders.text_loader import TextLoader
from apadata.text_processors.target_industries_utils import URL_PRESELECTION_TAGS_EN

from ..sitemap_api import SitemapAPI


@dataclass
class TestResponse:
    content: bytes


class TestResponseFailure:
    """Test Response"""

    def __init__(self, status_code, content):
        self.status_code = status_code
        self.content = content

    def json(self):
        return self.content


def mocked_get_response(endpoint: str) -> Optional[TestResponse]:
    if endpoint == "sitemap.xml":
        return TestResponse(
            content=bytes(
                TextLoader(
                    filepath="api/sitemap_api/tests/mocked_sitemap_xml_response.txt"
                ).load(),
                "utf-8",
            )
        )
    elif endpoint == "sitemap_index.xml":
        return TestResponse(
            content=bytes(
                TextLoader(
                    filepath="api/sitemap_api/tests/"
                    + "mocked_sitemap_index_xml_response.txt"
                ).load(),
                "utf-8",
            )
        )
    elif endpoint in URL_PRESELECTION_TAGS_EN:
        return TestResponse(
            content=bytes(
                TextLoader(
                    filepath="api/sitemap_api/tests/"
                    + "mocked_sitemap_index_xml_response.txt"
                ).load(),
                "utf-8",
            )
        )

    return None


def mocked_get_response_failure_requests(
    endpoint: str,
) -> Optional[RequestException]:
    if endpoint == "sitemap.xml":
        raise RequestException()
    elif endpoint == "sitemap_index.xml":
        raise RequestException()
    elif endpoint in URL_PRESELECTION_TAGS_EN:
        raise RequestException()
    return None


def mocked_get_response_failure_xml(
    endpoint: str,
) -> Optional[XMLSyntaxError]:
    if endpoint == "sitemap.xml":
        raise XMLSyntaxError()
    elif endpoint == "sitemap_index.xml":
        raise XMLSyntaxError()
    elif endpoint in URL_PRESELECTION_TAGS_EN:
        raise XMLSyntaxError()
    return None


@patch("apadata.api.sitemap_api.sitemap_api.SitemapAPI.get")
def test_sitemap_api(mock_get, snapshot):
    mock_get.side_effect = mocked_get_response
    domain = "https://kpmg.us"
    sitemap_api = SitemapAPI(domain=domain, sitemap=[])
    sitemap_api.determine_sitemap()
    sitemap_api.categorize_sitemap()

    assert len(sitemap_api.sitemap) == 5
    snapshot.assert_match(json.dumps(sitemap_api.sitemap), "sitemap-api-sitemap")
    snapshot.assert_match(
        json.dumps(sitemap_api.categories_to_urls),
        "sitemap-api-category-to-urls-success",
    )


@patch("apadata.api.sitemap_api.sitemap_api.SitemapAPI.get")
def test_sitemap_api_failure_requests(mock_get):
    mock_get.side_effect = mocked_get_response_failure_requests
    domain = "https://kpmg.us"
    sitemap_api = SitemapAPI(domain=domain, sitemap=[])
    sitemap_api.determine_sitemap()
    sitemap_api.categorize_sitemap()

    assert len(sitemap_api.sitemap) == 0
    assert sitemap_api.sitemap == []
    assert sitemap_api.categories_to_urls == {}


@patch("apadata.api.sitemap_api.sitemap_api.fromstring")
def test_sitemap_api_failure_xml(mock_get):
    mock_get.side_effect = mocked_get_response_failure_xml
    domain = "https://kpmg.us"
    sitemap_api = SitemapAPI(domain=domain, sitemap=[])
    sitemap_api.determine_sitemap()
    sitemap_api.categorize_sitemap()

    assert len(sitemap_api.sitemap) == 0
    assert sitemap_api.sitemap == []
    assert sitemap_api.categories_to_urls == {}
