from typing import Any

import os

import spacy

from .spacy_configuration import SpacyConfiguration


class Spacy:
    """
    Spacy builder class

    Parameters
    ----------
    nlp : Any
        Spacy pipeline
    config : SpacyConfiguration
        Configuration which contains all the required configurations for
        instantiating a spacy pipeline
    """

    nlp: Any = None

    def __init__(self, config: SpacyConfiguration):
        self.config = config
        self.load()

    def setup_extra_configs(self) -> Any:
        self.nlp.max_length = self.config.max_input_length

        if self.config.use_lookup_lemmatizer:
            self.nlp.remove_pipe(self.config.word_root_pipeline.value)
            self.nlp.add_pipe(
                self.config.word_root_pipeline.value, config={"mode": "lookup"}
            )

        if self.config.use_senter_over_parser:
            self.nlp.disable_pipe("parser")
            self.nlp.enable_pipe("senter")

        if (
            self.config.use_default_over_trainable
            and self.nlp.get_pipe(self.config.word_root_pipeline.value).is_trainable
        ):
            self.nlp.remove_pipe(self.config.word_root_pipeline.value)
            self.nlp.add_pipe(self.config.word_root_pipeline.value).initialize()

    @staticmethod
    def download(pipeline_name: str) -> Any:
        try:
            nlp = spacy.load(pipeline_name)
        except OSError:
            os.system(f"python -m spacy download {pipeline_name}")
            nlp = spacy.load(pipeline_name)
        return nlp

    def load(self) -> Any:
        self.nlp = Spacy.download(self.config.pipeline_name)

        self.setup_extra_configs()

        return self
